
 ; Final version A
; Trailing edge dimmer with touch and IR remote control.
; Frequency (50 or 60Hz) requires 220VAC supply
; zero voltage detection on one edge so at  each 20ms for 50Hz. Second half of waveform is calculated to produce phase control
; mains frequency is measured at power up for 50 or 60Hz operation

	ERRORLEVEL -302
	ERRORLEVEL -306

	    list      p=12F617        	; list directive to define processor
     #include <p12F617.inc>        ; processor specific variable definitions


   __CONFIG   _CP_OFF & _BOR_OFF & _MCLRE_OFF & _WDT_OFF & _PWRTE_ON &_INTRC_OSC_CLKOUT & _IOSCFS_8MHZ &_WRT_OFF

; RAM 
msCOUNT_LS	equ	H'20'	; 110ms counter ls byte
msCOUNT_MS	equ	H'21'	; 110ms counter ms byte
INT_FLG			equ	H'22'	;  flag to indicate 50us interrupt
STORE1			equ	H'23'	; delay counter	
STORE2			equ	H'24'	; delay counter
GATE_OFF0		equ	H'25'	; value in Timer1 to give required brightness ms byte
GATE_OFF1		equ	H'26'	; value in Timer1 to give brightness ls byte
CAL0			equ	H'27'	; calibration ms byte (ie count value of mains period)
CAL1			equ	H'28'	; calibration ls byte
TEMP			equ	H'29'	; temporary working
TEMP0			equ	H'2A'	; temporary ms byte
TEMP1			equ	H'2B'	; temporary ls byte
TEMP2			equ	H'2C'	; temporary ms byte
CYCLE_CAL		equ	H'2D'	; flags for calibration of cycle period
OFF			equ	H'2E'	; lamp off flag (clear is off)
CYC_COUNT	equ	H'2F'	; cycle counter
TEMPI			equ	H'30'	; temporary register used in interrupt
RATE			equ	H'31'	; dimming rate
TOUCHED		equ	H'32'	; touched plate flag
ALTERNATE		equ	H'33'	; up or down dimming flag
DIM_RUN		equ	H'34'	; dimming flag (dimming in process flag)
RPT				equ	H'35'	; startup repeat flag
PLATES			equ	H'36'	; plate touched or extension plate touched
MULT			equ	H'37'	; delay multiplier
DETECT		equ	H'38'	; touch detection counter
POSITION		equ	H'39'	; cycle position
MAX0			equ	H'3A'	; maximum zero to zero crossing half wave ms byte
MAX1			equ	H'3B'	; maximum zero to zero crossing half wave ls byte
DEBOUNCE		equ	H'3C'	; edge interrupt debounce
CALCON0		equ	H'3D'	; calculate gate on value for timer1 ms byte
CALCON1		equ	H'3E'	; calculate gate on value for timer1 ls byte
CALCOFF0		equ	H'40'	; calculate gate off value for timer1 ms byte
CALCOFF1		equ	H'41'	; calculate gate off value for timer1 ls byte
TMPI0			equ	H'42'	; temporary in interrupt ms byte
TMPI1			equ	H'43'	; temporary in interrupt ls byte
TEMPI0			equ	H'44'	; temporary in interrupt ms byte
NEUTRAL_Y_N	equ	H'45'	; initial startup delay value to set max brightness depending on if neutral is available
OFF_SET		equ	H'46'	; flag to clear OFF,0 when flag is set
ON_SET			equ	H'47'	; flag to set OFF,0 when flag is set
CONTINUOUS 	equ	H'48'	; continuous lamp on flag
VALUEI			equ	H'49'	; working value in interrupt
VALI0			equ	H'4A'	; working value in interrupt ms byte
VALI1			equ	H'4B'	; working value in interrupt ls byte
NEUTRL			equ	H'4C'	; neutral available for max brightness 
TEMP_VAL		equ	H'4D'	; temporary used in read 	
CAL_STO		equ	H'4E'	; using stored CAL flag
STORED0		equ	H'4F'	; stored ms byte
STORED1		equ	H'50'	; stored ls byte
EDGE_DET		equ	H'51'	; edge interrupt detect flag	

G_OFF_STORE0	equ	H'52'	; gate off position storage ms byte	
G_OFF_STORE1	equ	H'53'	; gate off position storage ls byte	
REQUIRED0		equ	H'54'	; required gate value when A,B or C used, ms byte
REQUIRED1		equ	H'55'	; required gate value when A,B or C used, ls byte	

STORAGE		equ	H'56'	; storage flag

; IR remote control decoding
COUNTER		equ	H'60'	; counter
REMOTE_A		equ	H'61'	; remote control Address byte
REM_A_BAR	equ	H'62'	; complementary value of address
REMOTE_C		equ	H'63'	; remote control Command byte
REM_C_BAR	equ	H'64'	; complementary value of command
BIT_COUNT		equ	H'65'	; 32 bit counter

; All banks RAM
W_TMP			equ	H'70'	; temporary store for w in interrupt
STATUS_TMP	equ	H'71'	; temporary store of status in interrupt 

; DATA (4 blocks)
DATA1ms		equ	H'78'	; data byte 1 ms 
DATA1ls			equ	H'79'	; data byte1 ls
DATA2ms		equ	H'7A'	; data 
DATA2ls			equ	H'7B'	; data
DATA3ms		equ	H'7C'	; data 
DATA3ls			equ	H'7D'	; data
DATA4ms		equ	H'7E'	; data 
DATA4ls			equ	H'7F'	; data
;___________________________________________________________
; initial values
	org	H'600'		; start address of where data memory is stored
; DATA1ms
	DE 	H'0B' 		; Initially set for when neutral available so use a maximum brightness setting and use value that's larger than the 4us count over 10ms
; DATA1ls
	DE	H'FF'		; 
;DATA2,3,4 not used but included in the 4 blocks of data
	
; ******************************************************************

; start at memory 0

	org		0				; reset vector
	goto	MAIN
; interrupt vector
	org		4
INTERRUPT
	movwf	W_TMP			; w to w_tmp storage
	swapf	STATUS,w		; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf		STATUS,RP0	; select memory bank 0

	btfsc	CYCLE_CAL	,1	; count waveform cycle for CAL, if clear get timer0 count from edge to edge
	goto	RUN

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

CALIBRATION	; at power up 

; calibrate values for period
	btfss	INTCON,INTF		; edge interrupt 
	goto	RECLAIM

	bcf		INTCON,INTF		; clear flag
; count cycles before start
	movf	CYC_COUNT,w
	btfsc	STATUS,Z
	goto	CAL_RUN
	decf	CYC_COUNT,f
	goto 	RECLAIM

CAL_RUN
; If CYCLE_CAL ,0 is clear then clear timer1 and start count with timer1
	btfss	CYCLE_CAL,0	; count waveform cycle for calibration
	goto	WAIT_NEXT_EDGE
; get timer1 count
	bcf		T1CON,0		; stop timer 1

; divide by 2
	bcf		STATUS,C		; carry clear
	rrf		TMR1H,f
	rrf		TMR1L,f
; store in CALC
	movf	TMR1H,w		; most significant store 
	movwf	CAL0
	movwf	MAX0
	movf	TMR1L,w		; least significant
	movwf	CAL1
	movwf	MAX1

FULL
; if stored value is > MAX, use MAX and restore in Flash

; get stored setting (READ)
	call		READ			; read stored data
	movwf	STORED0
	call		READ2
	movwf	STORED1

; compare

	movf	STORED0,w		; high byte
	subwf	MAX0,w			; subtraction
	movwf	VALUEI			; keep subtraction
	movf	STORED1,w		; low byte
	subwf	MAX1,w			; subtraction
	btfss	STATUS,C
	decf	VALUEI,f		; reduce high byte subtraction if carry	
	btfss	VALUEI,7
	goto	REDUCE		; less so keep stored

; place into stored Flash memory (WRITE)
	movf	MAX0,w
	movwf	DATA1ls
	movf	MAX1,w
	movwf	DATA2ls
	call 		WRITE			; write to memory

REDUCE
; reduce CAL value to account for phase lag in zero voltage crossing filter (4us per count)
	movlw	H'FF'			; low byte phase 
	subwf	CAL1,w
	movwf	VALI1
	movlw	D'0'				; high byte
	btfss	STATUS,C		; if carry then decrease ms byte
	addlw	D'1'
	subwf	CAL0,w
	movwf	VALI0
	clrf		NEUTRL			; neutral

TIMER_START
	bsf		T1CON,0		; start timer1 again
; enable interrupts
	bsf		STATUS,RP0	; select memory bank 1
	bsf		PIE1,TMR1IE		; timer 1 overflow interrupt
	bsf		PIE1,TMR2IE		; timer 2 match with PR2 enable
	bcf		STATUS,RP0	; select memory bank 0
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag
	bcf		PIR1,TMR2IF
	bsf		INTCON,PEIE	; peripheral interrupts enable
	bsf		CYCLE_CAL,1	; enable normal run
	goto	RECLAIM
WAIT_NEXT_EDGE
	clrf 		TMR1L			; cleared on edge
	clrf		TMR1H
	bsf		CYCLE_CAL,0	; set bit 0
	goto	RECLAIM

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

RUN	
; check interrupt
 ; CHECK TIMER2
	btfss	PIR1,TMR2IF		; timer 2, PR2 match, for 50us	 
	goto	EDGE

; clear timer2 match flag TMR2IF in PIR1
	bcf		PIR1,TMR2IF
	bsf		INT_FLG,0		; set flag to indicate a 50us period interrupt

;  delay for 
	movf	DEBOUNCE,w
	btfsc	STATUS,Z		; if zero bypass
	goto	PLATE_TOUCH
	
	decfsz	DEBOUNCE,f
	goto	PLATE_TOUCH
	bcf		INTCON,INTF
	bsf		INTCON,INTE	; reallow edge interrupt	

PLATE_TOUCH
; check GPIO,1, if low then plate touched
	btfss	GPIO,1
	bsf		DETECT,0	; detection flag

;  also check extension at GP0. if high then set detection flag

; check GPIO,0, if high then extension plate touched
	btfsc	GPIO,0
	bsf		DETECT,0	; detection flag

INC_ms
; increase 110ms repeat counter used for IR coding repeat function
	incfsz	msCOUNT_LS,f	; increase 110ms counter ls byte
	goto	EDGE
	incf		msCOUNT_MS,f	; increase 110ms counter ms byte

; edge detect
EDGE	
	btfss	INTCON,INTF		;  interrupt at zero crossing
	goto	GATE
	bcf		INTCON,INTF		; clear flag
	clrf		POSITION		; position in cycle starts at 0 each zero crossing
	clrf		EDGE_DET

; check if ON_SET  flag is set
	btfss	ON_SET,0		; when set, set OFF,0
	goto	OFFSET
	bsf		OFF,0
	clrf		ON_SET		; cleared when OFF flag is set
	goto	INIT_TM
OFFSET; check if OFFSET flag is set
	btfss	OFF_SET,0		; when set, set OFF,0
	goto	INIT_TM
	clrf		OFF			
	clrf		OFF_SET		; cleared when OFF flag is set
INIT_TM
; Initial timer for setting of max brightness
	movf	NEUTRAL_Y_N,w 	; initial startup delay value to set max brightness depending on if neutral is available
	btfsc	STATUS,Z			; when zero, bypass decrement
	goto	CYC_CNTR
; no decrement of NEUTRAL_Y_N when dimming
; check touch plate dimming

	btfss	DIM_RUN,0		; if set, restore NEUTRAL_Y_N to FF
	goto	REM_DIM
	movlw	H'FF'			; sets 5s
	movwf	NEUTRAL_Y_N	; initial startup delay value to set max brightness depending on if neutral is available	
	goto	CYC_CNTR		; dimming so no decrement
; check remote control dimming
REM_DIM
	btfss	RPT,0			; if set then restore NEUTRAL_Y_N to FF
	goto	DEC_N
	movlw	H'FF'			; sets 5s
	movwf	NEUTRAL_Y_N	; initial startup delay value to set max brightness depending on if neutral is available	
	goto	CYC_CNTR		; dimming so no decrement
DEC_N
	decf	NEUTRAL_Y_N,f	; both flags clear so decrement

CYC_CNTR
; cycle counter for touch plate delay
	btfsc	TOUCHED,0			; if clear, bypass increasing of cycle count
	incf		CYC_COUNT,f

; set up timer1 for when to switch off gate
G_ON
	bcf		GPIO,5
	movf	OFF,w	
	btfss	STATUS,Z		; if zero, then keep gate off
	bsf		GPIO,5			; gate on at zero crossing

	bcf		INTCON,INTE	; disable edge interrupt	 until after last gate off in the full cycle
	bcf		INTCON,INTF		; clear flag

; set plates
	btfsc	DETECT,0		; if touch detected, set plates flag
	goto	TCD			; detected
	clrf		PLATES
	goto	OUT
TCD
	bsf		PLATES,0	
	clrf		DETECT
OUT

;  if OFF use a bright level for zero voltage triggering calculation (without switching on lights)

	movf	OFF,w
	btfsc	STATUS,Z
	goto	CALC_OFF

 	btfsc	NEUTRL,0		; if clear, check if GATE_OFF0 and GATE_OFF1  is = or greater than CAL0,CAL1 then set CONTINUOUS flag
	goto	STANDARD
	clrf		CONTINUOUS

; compare with CAL (now in VALI0 and VALI1)
	movf	VALI0,w			; high byte
	subwf	GATE_OFF0,w	; subtraction
	movwf	VALUEI			; keep subtraction
	
	movf	VALI1,w			; low byte
	subwf	GATE_OFF1,w	; subtraction
	btfss	STATUS,C
	decf	VALUEI,f		; reduce high byte subtraction if carry	

	iorwf	VALUEI,w		; check if subtraction = 0 (ie the same values)
	btfsc	STATUS,Z
	bsf		CONTINUOUS,0	; equals gate off
	btfss	VALUEI,7
	bsf		CONTINUOUS,0	; CAL is less than GATE_OFF

STANDARD
; Calculate gate on and gate off positions
; preload timer. timer1  is 65536 minus GATE_OFF
	clrf		TMR1L 			; Clear Low byte, Ensures no rollover into TMR1H
	bcf		T1CON,TMR1ON	; timer off
	movf	GATE_OFF1,w	; low byte
	sublw	H'FF'			; low byte subtract
	movwf	TMR1L			; timer 1 low byte
	movf	GATE_OFF0,w	; high byte
	btfss   	STATUS,C    	 	; carry check
    	addlw   	D'1'       		 	; add one if carry
	sublw	H'FF'			; w has high byte subtraction
	movwf	TMR1H
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared
	bsf		T1CON,TMR1ON	; timer on
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared

; calculate for gate on and store ready for use
; GATE_OFF - MAX
	movf	MAX1,w			; low byte
	subwf	GATE_OFF1	,w	; low byte subtract
	movwf	CALCON1		; calculated on value for timer 1 low byte
	movf	MAX0,w			; high byte
	btfss   	STATUS,C    	 	; carry check
    	addlw   	D'1'       		 	; add one if carry
	subwf	GATE_OFF0,w	; w has high byte subtraction
	movwf	CALCON0

 ; Calculate for gate off and store
; value for timer1  is 65536 minus GATE_OFF
	movf	GATE_OFF1,w	; low byte
	sublw	H'FF'			; low byte subtract
	movwf	CALCOFF1		; calculated off value for timer 1 low byte
	movf	GATE_OFF0,w	; high byte
	btfss   	STATUS,C    	 	; carry check
    	addlw   	D'1'       		 	; add one if carry
	sublw	H'FF'			; w has high byte subtraction
	movwf	CALCOFF0
	goto	GATE

CALC_OFF
; get BRIGHT value
; divide CALibration value (full half cycle period)  by two. 
	bcf		STATUS,C
	rrf		CAL0,w
	movwf	TMPI0
	rrf		CAL1,w
	movwf	TMPI1
; Then by two again and add the two  for high brightness
	bcf		STATUS,C
	rrf		TMPI0,w
	movwf	TEMPI0
	rrf		TMPI1,w

; add
	addwf	TMPI1,f			; add ls bytes
	btfss	STATUS,C		; check for carry
	goto	NCI				; no carry
	movlw	D'1'
	addwf	TMPI0,f
NCI	movf	TEMPI0,w		; ms byte
	addwf	TMPI0,f			; ms byte

; Calculate gate on and gate off positions even though gate remains off
; preload timer. timer1  is 65536 minus GATE_OFF
; gate off period calculation
	clrf		TMR1L 			; Clear Low byte, Ensures no rollover into TMR1H
	bcf		T1CON,TMR1ON	; timer off
	movf	TMPI1,w			; low byte
	sublw	H'FF'			; low byte subtract
	movwf	TMR1L			; timer 1 low byte
	movf	TMPI0,w			; high byte
	btfss   	STATUS,C    	 	; carry check
    	addlw   	D'1'       		 	; add one if carry
	sublw	H'FF'			; w has high byte subtraction
	movwf	TMR1H
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared
	bsf		T1CON,TMR1ON	; timer on
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared

; calculate for gate on and store ready for use
; GATE_OFF - MAX
	movf	MAX1,w			; low byte
	subwf	TMPI1	,w		; low byte subtract
	movwf	CALCON1		; calculated on value for timer 1 low byte
	movf	MAX0,w			; high byte
	btfss   	STATUS,C    	 	; carry check
    	addlw   	D'1'       		 	; add one if carry
	subwf	TMPI0,w			; w has high byte subtraction
	movwf	CALCON0

 ; Calculate for gate off and store
; value for timer1  is 65536 minus GATE_OFF
	movf	TMPI1,w			; low byte
	sublw	H'FF'			; low byte subtract
	movwf	CALCOFF1		; calculated off value for timer 1 low byte
	movf	TMPI0,w			; high byte
	btfss   	STATUS,C    	 	; carry check
    	addlw   	D'1'       		 	; add one if carry
	sublw	H'FF'			; w has high byte subtraction
	movwf	CALCOFF0

GATE
; gate control with timer1 interrupt
	btfss	PIR1,TMR1IF		; timer 1 interrupt flag
	goto	RECLAIM

; check position in cycle. If POSITION is zero then switch gate off. If bit 0 is set then gate on, if bit 1 is set then gate off   
	movf	POSITION,w
	btfss	STATUS,Z
	goto	POS_ON
	bcf		PIR1,TMR1IF
	btfss	CONTINUOUS,0	; keep gate on if continuous flag set
	bcf		GPIO,5			; gate off 

; reload timer 1 for gate on

	clrf		TMR1L 			; Clear Low byte, Ensures no rollover into TMR1H
	bcf		T1CON,TMR1ON	; timer off

	movf	CALCON0,w		; calculated value for next gate on
	movwf	TMR1H
	movf	CALCON1,w
	movwf	TMR1L
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared
	bsf		T1CON,TMR1ON	; timer on
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared
	bsf		POSITION,0
	goto	RECLAIM

POS_ON
; check position
	btfss	POSITION,0
	goto	POS
; gate on	
	btfss	CONTINUOUS,0	; keep gate on if continuous flag set
	bcf		GPIO,5
	movf	OFF,w	
	btfss	STATUS,Z		; if zero, then keep gate off		
	bsf		GPIO,5

; preload timer for gate off. 
	clrf		TMR1L 			; Clear Low byte, Ensures no rollover into TMR1H
	bcf		T1CON,TMR1ON	; timer off

	movf	CALCOFF0,w	; calculated value for next gate off
	movwf	TMR1H
	movf	CALCOFF1,w
	movwf	TMR1L
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared
	bsf		T1CON,TMR1ON	; timer on
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared
	bsf		POSITION,1		; next position
	bcf		POSITION,0
	goto	RECLAIM

POS
	btfss	CONTINUOUS,0	; keep gate on if continuous flag set
	bcf		GPIO,5			; gate off 
	bcf		T1CON,TMR1ON	; timer off
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared
	clrf		POSITION		; return to 0

; debounce added 
	movlw	D'10'			; debounce 500us
	movwf	DEBOUNCE		; set delay to re-allow edge interrupt

; end of interrupt reclaim w and status 
RECLAIM
	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS			; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf  	W_TMP,w		; swap bits and into w register
	retfie					; return from interrupt

; ***********************************************************
		
MAIN
;  set oscillator calibration
	bsf		STATUS,RP0     	; bank 1
        movlw   D'0'     			 ; set oscillator to factory calibrated frequency 
        movwf   OSCTUNE
	bcf		STATUS,RP0
SETUP
; set inputs/outputs
	movlw	B'00000000'
	movwf	GPIO			; ports low/high. Mosfets off
	movlw	B'00000111'		; comparators off
	movwf	CMCON0
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000000'		; pullups off/on
	movwf	WPU
	movlw	B'00001111'		; outputs/inputs set 
	movwf	TRISIO			; port data direction register
	movlw	B'01000111'		; settings (pullups) prescaler/256 , bit 6 is for interrupt edge polarity
	movwf	OPTION_REG
; analog/ digital inputs
	movlw	B'00000000'		; all digital 
	movwf	ANSEL
	movlw	D'100'
	movwf	PR2				; timer2 match value for 50us
	movlw	B'00000000'		; 
	movwf	IOC				; interrupt on change for none
	bcf		STATUS,RP0	; select memory bank 0
;Timer1 on
	movlw 	B'00110001'		; divide by 8 for 4us per count
	movwf	T1CON
;Timer 2 on
	movlw	B'00000100'		; /1 prescaler /1 post scaler PR2 set to 100 for 50us interrupt
	movwf	T2CON

;  initial states
	clrf		GATE_OFF0		; brightness setting 
	clrf		GATE_OFF1		; brightness setting 
	clrf		TMR1L			; timer1 clear
	clrf		TMR1H
	clrf		OFF			; lamp off flag
	clrf		CYCLE_CAL		; count waveform cycle 
	movlw	D'50'			; 1s for 50Hz (0.833s for 60Hz)
	movwf	CYC_COUNT	; cycle counter. start of calibration after reduces to 0
	clrf		TOUCHED		; touched plate flag
	clrf		ALTERNATE		; up down dimming flag for touch control starts with dim up
	clrf		DIM_RUN		; dimming flag off (dimming in process flag)
	clrf		PLATES			; touched plate flag
	clrf		POSITION		; cycle position
	clrf		DEBOUNCE		; edge interrupt debounce
	clrf		CONTINUOUS 	; lamp on continuously flag
	clrf		NEUTRL			; neutral available for max brightness 
	clrf		CAL_STO		; using stored CAL flag
	movlw	H'FF'			; sets 5s
	movwf	NEUTRAL_Y_N	; initial startup delay value to set max brightness depending on if neutral is available
	clrf		STORAGE		; storage flag

; Startup timer
	movlw	D'133'			; 4s
	movwf	TEMP
STRT_DEL
	movlw	D'200'
	call		DY				; delay 
	decfsz	TEMP,f
	goto	STRT_DEL		; startup delay

; Interrupts
	bcf		INTCON,INTF
	bsf		INTCON,INTE	; edge interrupt
	bsf 		INTCON,GIE		; set global interrupt enable 
;
;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
;Test for touched plate
PLATE
; check if using stored CAL
	btfsc	CAL_STO,0		; using stored CAL flag
	goto	BYPASS 	
; if NEUTRAL_Y_N is clear use stored values for CAL,
	movf	NEUTRAL_Y_N,w
	btfss	STATUS,Z
	goto	BYPASS		; otherwise  if NEUTRAL_Y_N is >0 ie during initial stages where brightness level can be set, use Full values for CAL
; get CAL setting
	call		READ			; read stored data
	movwf	CAL0
	call		READ2
	movwf	CAL1
	bsf		CAL_STO,0		; using stored CAL flag

; if CAL >= to  VALI0,VALI1 then clear NEUTRL,0, otherwise set.
	bsf		NEUTRL,0		; initial
	movf	VALI0,w			; high byte
	subwf	CAL0,w			; subtraction
	movwf	VALUEI			; keep subtraction
	
	movf	VALI1,w			; low byte
	subwf	CAL1,w			; subtraction
	btfss	STATUS,C
	decf	VALUEI,f		; reduce high byte subtraction if carry	

	iorwf	VALUEI,w		; check if subtraction = 0 (ie the same values)
	btfsc	STATUS,Z
	clrf		NEUTRL			; equals gate off
	btfss	VALUEI,7
	clrf		NEUTRL			;  greater than 
BYPASS

	btfsc	PLATES,0		; if clear then plate is either indicated as having been touched or has not been touched
	goto	TOUCH_DETECTED

; count number of cycle times is detected for touch input (20ms per count)
	movf	CYC_COUNT,w	; read cycle counter (increased on port change interrupt at zero crossing)
; if 3 counts (60ms) or less then nothing happens 
	sublw	D'4'
	btfsc	STATUS,C		; if positive (Carry is set) then no effect
	goto	OFF_PLATE 
; if 16 counts or less  (320ms) then on/off
	movf	CYC_COUNT,w	; read cycle counter (increased on port change 'edge' interrupt at zero crossing)
	sublw	D'16'
	btfss	STATUS,C		; if positive (Carry is set) then on/off
	goto	OFF_PLATE		;

; switch on at max brightness setting or off (alternate action)	
; on and off control
	clrf		DIM_RUN		; dimming flag off (dimming in process flag)
	clrf		CYC_COUNT
	clrf		TOUCHED		; clear touched flag if plate not touched
	goto	OPERATE		; operate has lamp on at maximum brightness

OFF_PLATE
	clrf		CYC_COUNT
	clrf		DIM_RUN		; dimming flag off (dimming in process flag)
	clrf		TOUCHED		; clear touched flag if plate not touched
	goto	IR_IN

; Touch plate
TOUCH_DETECTED

; if more that 16 counts, then start dimming up or down 
	movf	CYC_COUNT,w	; read cycle counter (increased interrupt at zero crossing)
	sublw	D'16'
	btfsc	STATUS,C
	goto	INDICATE_TOUCHED
	clrf		TOUCHED		; stop cycle count increase

; if dimming flag is set bypass toggle alternate
	btfsc	DIM_RUN,0
	goto	TEST_ALT
; set dimming flag
	bsf		DIM_RUN,0
; toggle alternate flag	
	
	incf		ALTERNATE,f	; toggle bit 0 high or low
TEST_ALT
	btfss	ALTERNATE,0	; if set dim up
	goto	DIM_DOWN

DIM_UP 
; increase GATE_OFF0,1 up to CAL0,1

; dim rate set by whether in calibration or not
; if NEUTRAL_Y_N is zero, use faster dim rate
	movf	NEUTRAL_Y_N,w
	movlw	D'1'			; slower dim
	btfsc	STATUS,Z
	movlw	D'6'			; faster
	movwf	RATE

; set STORAGE flag so value can be stored in flash memory if in calibration mode
	bsf		STORAGE,0

 ; lamp on
	bsf		ON_SET,0
; transfer GATE_OFF to working registers
	movf	GATE_OFF0,w
	movwf	TEMP0
	movf	GATE_OFF1,w
	movwf	TEMP1
; increase GATE_OFF0,1 up to CAL0,1
	movf	RATE,w			; 4us per value increase
	addwf	TEMP1,f
	btfsc	STATUS,C		; if carry increase ms byte
	incf		TEMP0,f
; compare with CAL
	movf	CAL0,w			; high byte
	subwf	TEMP0,w		; subtraction
	movwf	TEMP2			; keep subtraction
	
	movf	CAL1,w			; low byte
	subwf	TEMP1,w		; subtraction
	btfss	STATUS,C
	decf	TEMP2,f			; reduce high byte subtraction if carry	

	iorwf	TEMP2,w		; check if subtraction = 0 (ie the same values)
	btfsc	STATUS,Z
	goto	DIM_EQ_UP		; equals gate off
	btfsc	TEMP2,7
	goto	DIM_LESS_UP	; less than GATE_OFF
DIM_EQ_UP
; more than gate_off so set at gate off CAL value	
; transfer CAL to gate_off
	bcf		STATUS,GIE
	movf	CAL0,w
	movwf	GATE_OFF0
	movf	CAL1,w
	movwf	GATE_OFF1
	bsf		STATUS,GIE

;"""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""
; Option
	goto	DIM_UP_CYCLE		; dimming stops at each end. (when not remarked out)
;""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""

SWAP; dimming changes direction after delay
	movlw	D'150'				; wait 1s
	movwf	MULT
WAIT1
	call		DELAYms
	decfsz	MULT,f
	goto	WAIT1
	incf		ALTERNATE,f		; dimming up/dn changes
	goto	TEST_ALT

; use increased value
DIM_LESS_UP
; transfer TEMP to gate_off
	bcf		STATUS,GIE
	movf	TEMP0,w
	movwf	GATE_OFF0
	movf	TEMP1,w
	movwf	GATE_OFF1
	bsf		STATUS,GIE

DIM_UP_CYCLE
	call		DELAYms		; 20ms approx
	btfsc	PLATES,0		; if set still touched
	goto	DIM_UP
	goto	PLATE

DIM_DOWN

; dim rate set by whether in calibration or not
; if NEUTRAL_Y_N is zero, use faster dim rate
	movf	NEUTRAL_Y_N,w
	movlw	D'1'			; slower dim
	btfsc	STATUS,Z
	movlw	D'6'			; faster
	movwf	RATE

; transfer GATE_OFF to working registers
	movf	GATE_OFF0,w
	movwf	TEMP0
	movf	GATE_OFF1,w
	movwf	TEMP1
; decrease GATE_OFF0,1 
	movf	RATE,w			; 4us per value increase
	subwf	TEMP1,f
	btfsc	STATUS,C		; if carry decrease ms byte
	goto	MORE_DIM_DOWN
; carry	
; if TEMP0 is zero then stop reduction
	movf	TEMP0,w
	btfsc	STATUS,Z
	goto	GATE_0A
	decf	TEMP0,f
	goto	MORE_DIM_DOWN
GATE_0A
; more than gate_off so set at 0
	bcf		STATUS,GIE
	clrf		GATE_OFF0
	clrf		GATE_OFF1
	bsf		STATUS,GIE
	bsf		OFF_SET,0		; lamp off

;"""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""
; Option; remark in/out as appropriate
	goto	DIM_DN_CYCLE		; dimming stops at limit
;	goto	SWAP			; dimming changes direction after a delay
;""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""""
; use decreased value
MORE_DIM_DOWN
; transfer TEMP to gate_off
	bcf		STATUS,GIE
	movf	TEMP0,w
	movwf	GATE_OFF0
	movf	TEMP1,w
	movwf	GATE_OFF1
	bsf		STATUS,GIE

DIM_DN_CYCLE
	call		DELAYms		; 20ms approx
	btfsc	PLATES,0		; if set still touched
	goto	DIM_DOWN
	goto	PLATE

INDICATE_TOUCHED
; check cycle counter
	bsf		TOUCHED,0		; set when touched. Allows CYC_COUNT to increase each zero crossing		
	goto	PLATE

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
; infrared remote 
IR_IN
	clrf		DIM_RUN		; dimming flag off (dimming in process flag)
	call		START			; decode IR code
; on return w has 00, FF or A0
	xorlw	H'00'
	btfsc	STATUS,Z	
	goto	PLATE			; code not correct
	xorlw	H'FF'
	btfsc	STATUS,Z	
	goto	PLATE			; code not correct

DECODE ; received codes
;A
	movlw	H'1F'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	DIM				;  for a low brightness
;B
	movlw	H'1E'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	MEDIUM			;  meduium brightness
;C
	movlw	H'1A'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	BRIGHT			;  for a bright lamp
;^
	movlw	H'05'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	UP
;<
	movlw	H'08'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	DOWN1
;>
	movlw	H'01'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	UP1
;V
	movlw	H'00'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	DOWN
;O
	movlw	H'04'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	ON_OFF

; Operate or power button
	movlw	H'1B'
	xorwf	REMOTE_C,w
	btfss	STATUS,Z
	goto	PLATE			; no valid codes

OPERATE ; switches on at maximum brightness and off (toggle/alternate)
; check OFF for on or off
	movf	OFF,w
	btfss	STATUS,Z		; when zero lamp is off
	goto	LAMP_OFF1

; lamp at maximum
	bsf		ALTERNATE,0	; force next dimming direction
; set gate off values at 0
	bcf		STATUS,GIE
	clrf		GATE_OFF0
	clrf		GATE_OFF1
	bsf		STATUS,GIE
; lamp on
	bsf		ON_SET,0
;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
; check if in adjust for max brightness (if Neutral used) or less if no Neutral
	movf	NEUTRAL_Y_N,w	; if zero then bypass
	btfss	STATUS,Z
	goto	PLACE

; increase gate drive up to CAL

	movlw	D'120'			; ramp up rateof 120 gives 400ms ramp up
	movwf	RATE

; increase GATE_OFF0, GATE_OFF1 up to CAL0 CAL1
RAMP0
	bsf		EDGE_DET,0
EDGE0
	btfsc	EDGE_DET,0	; cleared in edge interrupt	
	goto	EDGE0
	movf	RATE,w			; 4us per value increase
	bcf		STATUS,GIE
	addwf	GATE_OFF1,f
	btfsc	STATUS,C		; if carry increase ms byte
	incf		GATE_OFF0,f
	bsf		STATUS,GIE

; compare with original GATE_OFF with CAL
	movf	GATE_OFF0,w	; high byte
	subwf	CAL0,w			; subtraction
	movwf	TEMP2			; keep subtraction
	
	movf	GATE_OFF1,w	; low byte
	subwf	CAL1,w			; subtraction
	btfss	STATUS,C
	decf	TEMP2,f			; reduce high byte subtraction if carry	

	iorwf	TEMP2,w		; check if subtraction = 0 (ie the same values)
	btfsc	STATUS,Z
	goto	PLATE			; equals original gate off position
	btfss	TEMP2,7
	goto	RAMP0			; less than GATE_OFF

; more than gate_off so set at gate off CAL value	
; transfer CAL to gate_off
	bcf		STATUS,GIE
	movf	CAL0,w
	movwf	GATE_OFF0
	movf	CAL1,w
	movwf	GATE_OFF1
	bsf		STATUS,GIE
	goto	PLATE

ON_OFF ; switches on at last brightness and off (toggle/alternate)
; check OFF for on or off
	movf	OFF,w
	btfss	STATUS,Z		; when OFF is zero, lamp is off. alternate on and off
	goto	LAMP_OFF

; lamp ramps up to last brightness used already set Gate_off values
; increase gate drive up to GATE_OFF0,1 
	
	movlw	D'120'			; ramp up rate of 120 gives 400ms ramp up period
	movwf	RATE

; transfer GATE_OFF to working registers
	movf	GATE_OFF0,w
	movwf	TEMP0
	movf	GATE_OFF1,w
	movwf	TEMP1

; if values are zero keep lamp off and goto PLATE
	movf	GATE_OFF0,w
	btfss	STATUS,Z
	goto	NOT0
	movf	GATE_OFF1,w
	btfsc	STATUS,Z
	goto	PLATE
NOT0
; set GATE_OFF to 0
	bcf		STATUS,GIE
	clrf		GATE_OFF0
	clrf		GATE_OFF1	
	bsf		STATUS,GIE

; lamp on
	bsf		ON_SET,0
; increase GATE_OFF0 GATE_OFF1 up to TEMP0 TEMP1
RAMP1
	bsf		EDGE_DET,0
EDGE1
	btfsc	EDGE_DET,0	; cleared in edge interrupt	
	goto	EDGE1
	movf	RATE,w			; 4us per value increase
	bcf		STATUS,GIE
	addwf	GATE_OFF1,f
	btfsc	STATUS,C		; if carry increase ms byte
	incf		GATE_OFF0,f
	bsf		STATUS,GIE

; compare with original GATE_OFF stored in TEMP0 and TEMP1
	movf	GATE_OFF0,w	; high byte
	subwf	TEMP0,w		; subtraction
	movwf	TEMP2			; keep subtraction
	
	movf	GATE_OFF1,w	; low byte
	subwf	TEMP1,w		; subtraction
	btfss	STATUS,C
	decf	TEMP2,f			; reduce high byte subtraction if carry	

	iorwf	TEMP2,w		; check if subtraction = 0 (ie the same values)
	btfsc	STATUS,Z
	goto	END_RAMP1	; equals original gate off position
	btfss	TEMP2,7
	goto	RAMP1			; less than GATE_OFF

; more than original GATE_OFF so set at original value
; transfer CAL to gate_off
	bcf		STATUS,GIE
	movf	TEMP0,w
	movwf	GATE_OFF0
	movf	TEMP1,w
	movwf	GATE_OFF1
	bsf		STATUS,GIE

END_RAMP1
	bcf		ALTERNATE,0	; force next dimming direction
	goto	PLATE

; ____________________________________________________________________________________

LAMP_OFF ; for on/off on remote control 'O' button
	call		RAMP_DOWN	; for an off, run a ramp down dimming subroutine

	goto	PLATE

LAMP_OFF1; for on/off for touch plate and Operate button on remote control
	call		RAMP_DOWN

	goto	CK_ADJ

;___________________________________________________________________________________
;subroutine
RAMP_DOWN ; for off, run a ramp down dimming

; store GATE_OFF0 and GATE_OFF1 to G_OFF_STORE0 and G_OFF_STORE1
	movf	GATE_OFF0,w
	movwf	G_OFF_STORE0
	movf	GATE_OFF1,w
	movwf	G_OFF_STORE1

; ramp GATE_OFF down to 0
	movlw	D'120'			; ramp rate
	movwf	RATE

; transfer GATE_OFF to working registers
	movf	GATE_OFF0,w
	movwf	TEMP0
	movf	GATE_OFF1,w
	movwf	TEMP1
; decrease GATE_OFF0,1 
DEC3
	bsf		EDGE_DET,0
EDGE3
	btfsc	EDGE_DET,0	; cleared in edge interrupt	
	goto	EDGE3
	movf	RATE,w			; 4us per value increase
	subwf	TEMP1,f
	btfsc	STATUS,C		; if carry decrease ms byte
	goto	MORE_DIM_DOWN3
; carry	
; if TEMP0 is zero then stop reduction
	movf	TEMP0,w
	btfsc	STATUS,Z
	goto	GATE_0A3
	decf	TEMP0,f
; use decreased value
MORE_DIM_DOWN3
; transfer TEMP to gate_off
	bcf		STATUS,GIE
	movf	TEMP0,w
	movwf	GATE_OFF0
	movf	TEMP1,w
	movwf	GATE_OFF1
	bsf		STATUS,GIE
	goto	DEC3

GATE_0A3
; more than gate_off so set at 0
	bcf		STATUS,GIE
	clrf		GATE_OFF0
	clrf		GATE_OFF1
	bsf		STATUS,GIE

; set lamp off	
	bsf		OFF_SET,0		; lamp off
; restore GATE_OFF values from G_OFF_STORE	
	movf	G_OFF_STORE0,w
	movwf	GATE_OFF0
	movf	G_OFF_STORE1,w
	movwf	GATE_OFF1
	bcf		ALTERNATE,0	; force next dimming direction
	return


;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
CK_ADJ
; check if in adjust for max brightness (if Neutral used) or less if no Neutral
	movf	NEUTRAL_Y_N,w	; if zero then bypass
	btfsc	STATUS,Z
	goto	PLACE
; check if STORAGE flag is set (meaning that dimming up was done during calibration)
	btfss	STORAGE,0
	goto	PLACE

; place GATE_OFF into DATA1 and DATA2 ready to write to memory
	movf	GATE_OFF0,w
	movwf	CAL0
	movwf	DATA1ls
	movf	GATE_OFF1,w
	movwf	CAL1
	movwf	DATA2ls
	call 		WRITE			; write to memory
	clrf		NEUTRAL_Y_N	; stop further changes to set max brightness depending on if neutral is available
PLACE
	clrf		NEUTRAL_Y_N	; end the calibration for max brightness 
	bcf		STATUS,GIE
	clrf		GATE_OFF0		; gate off to zero
	clrf		GATE_OFF1
	bsf		STATUS,GIE
	goto	PLATE	

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

DOWN1 ; faster rate
	movlw	D'150'
	goto	DOWN_RATE
DOWN ; decrease GATE_OFF0,1 down to 0
	movlw	D'25'
DOWN_RATE
	movwf	RATE
DOWN_RUN
; transfer GATE_OFF to working registers
	movf	GATE_OFF0,w
	movwf	TEMP0
	movf	GATE_OFF1,w
	movwf	TEMP1
; decrease GATE_OFF0,1 
	movf	RATE,w				; 4us per value increase
	subwf	TEMP1,f
	btfsc	STATUS,C		; if carry decrease ms byte
	goto	MORE_DOWN
; carry	
; if TEMP0 is zero then stop reduction
	movf	TEMP0,w
	btfsc	STATUS,Z
	goto	GATE_0
	decf	TEMP0,f
	goto	MORE_DOWN
GATE_0
; more than gate_off so set at 0
	bcf		STATUS,GIE
	clrf		GATE_OFF0
	clrf		GATE_OFF1
	bsf		STATUS,GIE
	bsf		OFF_SET,0		; lamp off
	goto	PLATE
; use decreased value
MORE_DOWN
; transfer TEMP to gate_off
	bcf		STATUS,GIE
	movf	TEMP0,w
	movwf	GATE_OFF0
	movf	TEMP1,w
	movwf	GATE_OFF1
	bsf		STATUS,GIE

REP_DOWN
	call		REPEAT
	xorlw	D'00'			; if zero repeat is off
	btfsc	STATUS,Z
	goto	PLATE
	goto	DOWN_RUN

UP1 ; faster rate
	movlw	D'150'
	goto	UP_RATE
UP 
; increase GATE_OFF0,1 up to CAL0,1
	movlw	D'25'
UP_RATE
	movwf	RATE

; set STORAGE flag so value can be stored in flash memory if s calibration mode
	bsf		STORAGE,0

 ; lamp on
	bsf		ON_SET,0
UP_RUN
; transfer GATE_OFF to working registers
	movf	GATE_OFF0,w
	movwf	TEMP0
	movf	GATE_OFF1,w
	movwf	TEMP1
; increase GATE_OFF0,1 up to CAL0,1
	movf	RATE,w			; 4us per value increase
	addwf	TEMP1,f
	btfsc	STATUS,C		; if carry increase ms byte
	incf		TEMP0,f
; compare with CAL
	movf	CAL0,w			; high byte
	subwf	TEMP0,w		; subtraction
	movwf	TEMP2			; keep subtraction
	
	movf	CAL1,w			; low byte
	subwf	TEMP1,w		; subtraction
	btfss	STATUS,C
	decf	TEMP2,f			; reduce high byte subtraction if carry	

	iorwf	TEMP2,w		; check if subtraction = 0 (ie the same values)
	btfsc	STATUS,Z
	goto	EQ_UP			; equals gate off
	btfsc	TEMP2,7
	goto	LESS_UP		; less than GATE_OFF
EQ_UP
; more than gate_off so set at gate off CAL value	
; transfer CAL to gate_off
	bcf		STATUS,GIE
	movf	CAL0,w
	movwf	GATE_OFF0
	movf	CAL1,w
	movwf	GATE_OFF1
	bsf		STATUS,GIE

	goto	REP_UP

; use increased value
LESS_UP
; transfer TEMP to gate_off
	bcf		STATUS,GIE
	movf	TEMP0,w
	movwf	GATE_OFF0
	movf	TEMP1,w
	movwf	GATE_OFF1
	bsf		STATUS,GIE

REP_UP
	call		REPEAT
	xorlw	D'00'		; if zero repeat is off
	btfsc	STATUS,Z
	goto	PLATE
	goto	UP_RUN

MEDIUM
; divide CALibration value (full half cycle period)  by two for mid setting brightness
;  ramp up GATE_OFF0,  GATE_OFF1 to Medium brightness value in REQUIRED0, REQUIRED1
	bcf		STATUS,C
	rrf		CAL0,w
	movwf	REQUIRED0
	rrf		CAL1,w
	movwf	REQUIRED1

; RAMPUP 
; check OFF for on or off
	movf	OFF,w
	btfss	STATUS,Z		; when OFF is zero, lamp is off.
	goto	COMPARE		; check if current GATE_OFF value is more or less than required

;_____________________________________________________________________
; lamp ramps up from zero
LAMP_PRESET

; set GATE_OFF to 0
	bcf		STATUS,GIE
	clrf		GATE_OFF0
	clrf		GATE_OFF1	
	bsf		STATUS,GIE

LAMP_PRESET1		; lamp ramps up from current value
	movlw	D'120'			; ramp up rate of 120 gives 400ms ramp up period
	movwf	RATE

; transfer GATE_OFF to working registers
	movf	GATE_OFF0,w
	movwf	TEMP0
	movf	GATE_OFF1,w
	movwf	TEMP1
; lamp on
	bsf		ON_SET,0
; increase GATE_OFF0 GATE_OFF1 up to required brightness
RAMP_M_D_B ; medium, dim or bright
	bsf		EDGE_DET,0
EDGE_M_D_B
	btfsc	EDGE_DET,0	; cleared in edge interrupt	
	goto	EDGE_M_D_B
	movf	RATE,w			; 4us per value increase
	bcf		STATUS,GIE
	addwf	GATE_OFF1,f
	btfsc	STATUS,C		; if carry increase ms byte
	incf		GATE_OFF0,f
	bsf		STATUS,GIE

; compare  GATE_OFF with required 
	movf	GATE_OFF0,w	; high byte
	subwf	REQUIRED0,w		; subtraction
	movwf	TEMP2			; keep subtraction
	
	movf	GATE_OFF1,w	; low byte
	subwf	REQUIRED1,w	; subtraction
	btfss	STATUS,C
	decf	TEMP2,f			; reduce high byte subtraction if carry	

	iorwf	TEMP2,w		; check if subtraction = 0 (ie the same values)
	btfsc	STATUS,Z
	goto	PLATE			; equal in value 
	btfss	TEMP2,7
	goto	RAMP_M_D_B	; less than GATE_OFF

; more than original GATE_OFF so set at original value
; transfer REQUIRED to gate_off
	bcf		STATUS,GIE
	movf	REQUIRED0,w
	movwf	GATE_OFF0
	movf	REQUIRED1,w
	movwf	GATE_OFF1
	bsf		STATUS,GIE
	goto	PLATE

; end ramp up 
;_________________________________________________________________________

DIM	
; divide CALibration value (full half cycle period)  by four for low brightness
	bcf		STATUS,C
	rrf		CAL0,w
	movwf	REQUIRED0
	rrf		CAL1,w
	movwf	REQUIRED1

	bcf		STATUS,C
	rrf		REQUIRED0,f
	rrf		REQUIRED1,f
	
; RAMPUP 
; check OFF for on or off
	movf	OFF,w
	btfss	STATUS,Z		; when OFF is zero, lamp is off.
	goto	COMPARE		; check if current GATE_OFF value is more or less than required
; lamp ramps up from zero
	goto	LAMP_PRESET

BRIGHT
; divide CALibration value (full half cycle period)  by two. 
	bcf		STATUS,C
	rrf		CAL0,w
	movwf	REQUIRED0
	rrf		CAL1,w
	movwf	REQUIRED1
; Then by two again and add the two for high brightness
	bcf		STATUS,C
	rrf		REQUIRED0,w
	movwf	TEMP2
	rrf		REQUIRED1,w

; add
	addwf	REQUIRED1,f	; add ls bytes
	btfss	STATUS,C		; check for carry
	goto	NCR			; no carry
	movlw	D'1'
	addwf	TEMP2,f
NCR
	movf	TEMP2,w		; ms byte
	addwf	REQUIRED0,f	; ms byte

; RAMPUP 
; check OFF for on or off
	movf	OFF,w
	btfss	STATUS,Z		; when OFF is zero, lamp is off.
	goto	COMPARE		; check if current GATE_OFF value is more or less than required
; lamp ramps up from zero
	goto	LAMP_PRESET

COMPARE
; brightness of lamp is on 
; check if dimming value greater or less than required (this determines whether ramp up or down from current brightness)
; compare  GATE_OFF with required 
	movf	GATE_OFF0,w	; high byte
	subwf	REQUIRED0,w	; subtraction
	movwf	TEMP2			; keep subtraction
	
	movf	GATE_OFF1,w	; low byte
	subwf	REQUIRED1,w	; subtraction
	btfss	STATUS,C
	decf	TEMP2,f			; reduce high byte subtraction if carry	

	iorwf	TEMP2,w		; check if subtraction = 0 (ie the same values)
	btfsc	STATUS,Z
	goto	PLATE			; equal in value 
	btfss	TEMP2,7
; more than original GATE_OFF so dim up
	goto	LAMP_PRESET1


; dim down to required
; less than original GATE_OFF so dim down
; ramp GATE_OFF down 
	movlw	D'120'			; ramp rate
	movwf	RATE

; transfer GATE_OFF to working registers
	movf	GATE_OFF0,w
	movwf	TEMP0
	movf	GATE_OFF1,w
	movwf	TEMP1
; decrease GATE_OFF0,1 
DOWN5
	bsf		EDGE_DET,0
EDGE5
	btfsc	EDGE_DET,0	; cleared in edge interrupt	
	goto	EDGE5
	movf	RATE,w			; 4us per value increase
	subwf	TEMP1,f
	btfsc	STATUS,C		; if carry decrease ms byte
	goto	MORE_DIM_DOWN5
; carry	
; if TEMP0 is zero then stop reduction
	movf	TEMP0,w
	btfsc	STATUS,Z
	goto	DOWN6
	decf	TEMP0,f
; test if at required value ie TEMP= or < REQUIRED
MORE_DIM_DOWN5
; compare  GATE_OFF with required 
	movf	TEMP0,w		; high byte
	subwf	REQUIRED0,w	; subtraction
	movwf	TEMP2			; keep subtraction
	
	movf	TEMP1,w		; low byte
	subwf	REQUIRED1,w	; subtraction
	btfss	STATUS,C
	decf	TEMP2,f			; reduce high byte subtraction if carry	

	iorwf	TEMP2,w		; check if subtraction = 0 (ie the same values)
	btfsc	STATUS,Z
	goto	PLATE			; equal in value 
	btfsc	TEMP2,7
	goto	MORE_THAN
; less than original GATE_OFF so set at original value
	bcf		STATUS,GIE
	movf	REQUIRED0,w
	movwf	GATE_OFF0	
	movf	REQUIRED1,w
	movwf	GATE_OFF1	
	bsf		STATUS,GIE
	goto	PLATE
	
MORE_THAN
; use decreased value
; transfer TEMP to gate_off
	bcf		STATUS,GIE
	movf	TEMP0,w
	movwf	GATE_OFF0
	movf	TEMP1,w
	movwf	GATE_OFF1
	bsf		STATUS,GIE
	goto	DOWN5

DOWN6
; more than gate_off so set at 0
	bcf		STATUS,GIE
	clrf		GATE_OFF0
	clrf		GATE_OFF1
	bsf		STATUS,GIE
	goto	PLATE

; ************************************************
; IR subroutines

; read IR signal that's based on pulse distance protocol
; start train is a 38kHz burst at 9ms and 4.5ms of no burst
; data sent as an 8-bit address (LSB first) then a complementary version of the address
; further data sent as an 8-bit command (LSB first) then a complementary version of the command
; logic 1 is 560us 38kHz burst and 1690us of no burst
; logic 0 is a 560us 38kHz burst and 560us of no burst
; At the end of the 32bit data is a tail pulse comprising 560us of a 38kHz burst
; data is sent once
; a repeat indication (transmit button held on) is 9000us 38kHz burst followed by 2250us no burst and a 560us 38kHz burst
; this is repeated each 110ms
; **********************************


START ; start of Infra Red reception

	clrf 		RPT			; initial startup 

; Check for start train
; comprises a 9ms 38kHz burst, read as a low of 9ms on IR detector, then followed by a high of 4.5ms (no 38kHz signal)
; count 180 x 50us interrupts where GPIO,3 needs to stay low
; 
	movlw	D'160'		; allow an 8ms minimum 
	movwf	COUNTER
; check PORT
CK_3
	btfsc	GPIO,3
	retlw 	H'00'		; if port goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3
; GPIO,3 can now go high up to a period of 3ms
; wait for more interrupts
	movlw	D'60'		; 3ms
	movwf	COUNTER
; GPIO,3 can go high in this period
CK_3_1
	btfsc	GPIO,3
	goto	NINE_ms	; GPIO,3  has gone high within the acceptable ~9ms low period (8-11ms window)
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_1
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_1
	retlw 	H'FF' 		; if GPIO,3 stays low  

NINE_ms	; GPIO,3 has stayed low and then gone high within the acceptable ~9ms low period (8-11ms window)
; Check for 4.5ms high
	movlw	D'70'		; 3.5ms
	movwf	COUNTER
; check GPIO,3

CK_3_2
	btfss	GPIO,3
	retlw	H'FF'		; if GPIO,3 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_2 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_2

; GPIO,3 can now go low up to a period of 2ms
; wait for more interrupts
	movlw	D'40'		; 2ms
	movwf	COUNTER
; GPIO,3 can go low in this period
CK_3_3
	btfss	GPIO,3
	goto	FOUR_POINT_FIVE_ms	; GPIO,3 has gone low within the acceptable ~4.5ms high period (3.5-5.5ms window)
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_3
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_3
	retlw	H'00'		; GPIO,3 stays high

; end of start train check
FOUR_POINT_FIVE_ms

; start 110ms counter at beginning of code
	clrf	msCOUNT_MS	; 110ms counter
	clrf	msCOUNT_LS
	
	movlw	D'32'		; set up 32bit counter
	movwf	BIT_COUNT

START_DATA ; start of data 
; clear data
	clrf	REMOTE_A		; remote control Address byte
	clrf	REM_A_BAR	; complementary value of address
	clrf	REMOTE_C		; remote control Command byte
	clrf	REM_C_BAR	; complementary value of command

; start reading data
READ_DATA
; a 560us low
	movlw	D'8'			; allow a 400us minimum 
	movwf	COUNTER
; check GPIO,3
CK_3_4
	btfsc	GPIO,3
	retlw 	H'00'		; if GPIO,3 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_4 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_4
; GPIO,3 can now go high up to a period of 400us for a 800us maximum
; wait for more interrupts
	movlw	D'8'			; 400us
	movwf	COUNTER
; GPIO,3 can go high in this period
CK_3_5
	btfsc	GPIO,3
	goto	ONE_ZERO	; GPIO,3 has gone high within the acceptable ~560us low period (400us-800us window)
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_5
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_5
	retlw 	H'FF' 		; GPIO,3 stays low  
; test for a zero (high for 560us) or a one (high for 1690us)
ONE_ZERO
; Check for 560us high
	movlw	D'8'			; allow a 400us minimum 
	movwf	COUNTER
; check GPIO,3

CK_3_6
	btfss	GPIO,3
	retlw	H'FF'		; if GPIO,3 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_6 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_6

; GPIO,3 can now go high up to a period of 400us for a 800us maximum
; wait for more interrupts
	movlw	D'8'			; 400us
	movwf	COUNTER

; GPIO,3 can go low in this period
CK_3_7
	btfss	GPIO,3
	goto	LOGIC_0	; GPIO,3 has gone low within the acceptable ~560us high period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_7
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_7
	goto	CHECK_FOR_1

LOGIC_0	; logic zero detected
	bcf		STATUS,C	; clear carry
; shift detected data into storage
SHIFT_DATA	
	rrf		REM_C_BAR,f	; complementary value of command
	rrf		REMOTE_C,f	; remote control Command byte
	rrf		REM_A_BAR,f	; complementary value of address
	rrf		REMOTE_A,f		; remote control Address byte
	
	decfsz	BIT_COUNT,f		; shift data for 32 bits
	goto	READ_DATA	; contine reading data
	goto	TAIL			; end of 32 bit data, check for the tail (560us of burst)
	
CHECK_FOR_1 ; check for a logic 1

; Check for 1690us high  (already 560us high) so check for 1130us
	movlw	D'10'		; 500us (1300us min)
	movwf	COUNTER
; check GPIO,3

CK_3_8
	btfss	GPIO,3
	retlw	H'FF'		; if GPIO,3 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_8 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_8

; GPIO,3 can now go high up to a period of 600us extra
; wait for more interrupts

	movlw	D'14'		; 700us (2000us max)
	movwf	COUNTER

; GPIO,3 can go low in this period
CK_3_9
	btfss	GPIO,3
	goto	LOGIC_1	; GPIO,3 has gone low within the acceptable ~1690us high period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_9
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_9
	retlw	H'00'		; GPIO,3 stays high	

LOGIC_1
	bsf		STATUS,C	; data is a 1
	goto	SHIFT_DATA

TAIL ; end of data tail

; a 560us low
	movlw	D'8'			; allow a 400us minimum 
	movwf	COUNTER
; check GPIO,3
CK_3_10
	btfsc	GPIO,3
	retlw 	H'00'		; if GPIO,3 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_10 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_10
; GPIO,3 can now go high up to a period of 400us for a 800us maximum
; wait for more interrupts
	movlw	D'8'			; 400us
	movwf	COUNTER
; GPIO,3 can go high in this period
CK_3_11
	btfsc	GPIO,3
	goto	TAIL_DET	; GPIO,3 has gone high within the acceptable ~560us low period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_11
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_11
	retlw 	H'FF' 		; if GPIO,3 stays low  

TAIL_DET ; tail is detected

; check if Address and Address bar are complementary

	comf	REM_A_BAR,w	; complementary value of address
	xorwf	REMOTE_A,w	; remote control Address byte
	btfss	STATUS,Z
	retlw	H'00'			; GPIO,3 high

; ********************************************

; For the particular remote that uses 08 as address,
; check for a 08 to specify the IR transmitter used or change if different
; or remark out if for use with other IR remotes with different address
	movlw	H'08'
	xorwf	REMOTE_A,w	; value of address
	btfss	STATUS,Z
	goto	PLATE
; *******************************************
; check if Command and Command bar are complementary

	comf	REM_C_BAR,w	; complementary value of command
	xorwf	REMOTE_C,w	; remote control Command byte
	btfss	STATUS,Z
	goto	PLATE

	retlw	H'A0'		; if data is decoded correctly send an 'AO'k return message

;------------------------------------------------------------------------
REPEAT
; wait for a low after 110ms (see msCOUNT_MS msCOUNT_LS	; 110ms counter)
; a return with FF is a repeat, return with 0 is non repeat

; REPEAT is a 9ms burst followed by 2250us off and 560us burst repeated 110ms apart
; wait for mscount within a 110ms range

KEEP_CHK_COUNT
	movf	msCOUNT_MS,w
	btfsc	RPT,0				; if clear use H06
	goto	REPEAT1
	sublw	H'06'				; 76.8ms plus the initial 13.5ms start (90.3ms)
	goto	REPEAT2
REPEAT1
	sublw	H'07'				; 89.6ms
REPEAT2
	btfsc	STATUS,C
	goto	KEEP_CHK_COUNT

; wait for GPIO,3 to go low
KEEP_CHK_LOW
; check if ms count is less than A00 
	movf	msCOUNT_MS,w
	btfsc	RPT,0				; if clear use H0A
	goto	REPEAT3
	sublw	H'0A'				; 128ms plus 13.5ms = 141.5ms
	goto	REPEAT4
REPEAT3
	sublw	H'0B'				; 140.8ms
REPEAT4
	btfss	STATUS,C
	retlw	D'00'
	btfsc	GPIO,3				; when low check if within time frame allowed
	goto	KEEP_CHK_LOW

; start 110ms counter at beginning of repeat frame
	bsf		RPT,0				; repeat frame counter flag use different ms count depending on whether first repeat (starts counting  after initial 13.5ms start or next repeats where initial start 
; period is not included)
	clrf		msCOUNT_MS		; 110ms counter
	clrf		msCOUNT_LS
; Check for repeat train
; comprises a 9ms 38kHz burst, read as a low of 9ms on IR detector, then followed by a high of 2.25ms (no 38kHz signal) then a 560us burst
; count of around 180 x 50us interrupts where GPIO,3 needs to stay low
; 
	movlw	D'160'		; allow an 8ms minimum 
	movwf	COUNTER
; check GPIO,3
CK_3_12
	btfsc	GPIO,3
	retlw	D'00'		; if GPIO,3 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_12 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_12
; GPIO,3 can now go high up to a period of 2ms
; wait for more interrupts
	movlw	D'40'		; 2ms
	movwf	COUNTER
; GPIO,3 can go high in this period
CK_3_13
	btfsc	GPIO,3
	goto	NINE_ms_1	; GPIO,3 has gone high within the acceptable ~9ms low period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_13
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_13
	return

NINE_ms_1

; Check for 2.25ms high
	movlw	D'30'		; 1.5ms minimum
	movwf	COUNTER
; check GPIO,3

CK_3_14
	btfss	GPIO,3
	retlw	D'00'		; if GPIO,3 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_14 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_14

; GPIO,3 can now go low up to a period of 2ms
; wait for more interrupts
	movlw	D'30'		; 2ms
	movwf	COUNTER
; GPIO,3 can go low in this period
CK_3_15
	btfss	GPIO,3
	goto	TWOPOINTTWOFIVE_ms	; GPIO,3 has gone low within the acceptable ~2.25ms high period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_15
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_15
	return
TWOPOINTTWOFIVE_ms
; a 560us low
	movlw	D'8'			; allow a 400us minimum 
	movwf	COUNTER
; check GPIO,3
CK_3_16
	btfsc	GPIO,3
	retlw	D'00'		; if GPIO,3 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_16 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_16
; GPIO,3 can now go high up to a period of 400us for an 800us maximum
; wait for more interrupts
	movlw	D'8'			; 400us
	movwf	COUNTER
; GPIO,3 can go high in this period
CK_3_17
	btfsc	GPIO,3
	retlw	H'FF'		; GPIO,3 has gone high within the acceptable ~560us low period. Repeat detected
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_17
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_17
	retlw	D'00'		; no repeat


; *****************************************************************************************************************************************************
; delay loop 

DELAYms; approx 6.66ms
	movlw	D'23'		; delay value
DY	movwf	STORE1		; STORE1 is number of loops value
LOOP8	
	movlw	D'117'
	movwf	STORE2		; STORE2 is internal loop value	
LOOP9
	decfsz	STORE2,f
	goto	LOOP9
	decfsz	STORE1,f
	goto	LOOP8
	return

; read data memory
READ
	bsf		STATUS,RP0	; select memory bank 1 
	movlw 	H'06'
	movwf 	PMADRH		; ms Byte of Program Address to read
	movlw	H'00'
	movwf 	PMADRL 		; ls Byte of Program Address to read

	bsf	 	PMCON1,RD 	; Read
	nop 					
	nop
; memory is read in second cycle PM read instruction
	movf	PMDATH,w 		; ms Byte of Program data 
;	movwf	TEMP_VAL
	movf	PMDATL,w 		; ls Byte of Program data
	bcf		STATUS,RP0	; bank 0
	return
READ2
	bsf		STATUS,RP0	; select memory bank 1 
	movlw 	H'06'
	movwf 	PMADRH		; ms Byte of Program Address to read
	movlw	H'01'
	movwf 	PMADRL 		; ls Byte of Program Address to read

	bsf	 	PMCON1,RD 	; Read
	nop 					
	nop
; memory is read in second cycle PM read instruction
	movf	PMDATH,w 		; ms Byte of Program data 
;	movwf	TEMP_VAL
	movf	PMDATL,w 		; ls Byte of Program data
	bcf		STATUS,RP0	; bank 0
	return

; write to data memory
WRITE
; This write routine assumes the following:
	bcf		GPIO,5			; gate off
	bcf		 INTCON,GIE 	; Disable interrupts 

	bsf		STATUS,RP0	; select memory bank 1
	movlw 	H'06'
	movwf 	PMADRH		; MS Byte of Program Address to write
	movlw	H'00'
	movwf 	PMADRL 		; LS Byte of Program Address to write

	movlw	H'78'			; Load initial data address
	movwf	FSR 
LOOP_WRITE 
	movf	 INDF,w 			; Load first data byte into upper byte
	movwf	 PMDATH		;
	incf		 FSR,f 			; Next byte
	movf	 INDF,w 			; Load second data byte into lower byte
	movwf	 PMDATL 		; 		
	incf		 FSR,f 			;
	bsf		 PMCON1,WREN ; Enable writes

; Required Sequence
	movlw	H'55'			; Start of required write sequence:
	movwf	PMCON2		; Write 55h
	movlw	H'AA'		 	;
	movwf	PMCON2 		; Write AAh
	bsf		PMCON1,WR 	; Set WR bit to begin write
	nop					 	; NOPs required for time to transfer data to the buffer registers
	nop						; 
;
	bcf		PMCON1,WREN ; Disable writes
	movf	PMADRL,w
	incf		PMADRL,f		; Increment address
; select 4,8,12 or16 	
	; 0F = 16 words
	; 0B = 12 words
	; 07 =  8 words
	; 03 =  4 words
	andlw	H'03'			; Indicates when the set number of words have been programmed
	sublw	H'03'

	btfss	STATUS,Z 		; Exit on a match,
	goto	LOOP_WRITE	; Continue until ended
	bcf		STATUS,RP0	; bank 0
	bsf		 INTCON,GIE 	; enable interrupts 
	return

	end
